/*
----------------------------------------------------------------------------

 Copyright (C) Sartorius Stedim Data Analytics AB 2017 -

 Use, modification and distribution are subject to the Boost Software
 License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)

----------------------------------------------------------------------------
*/

#include "stdafx.h"
#include "sqpplusrunner.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "utf8util.h"

/* Functions to present the prediction result */
int SQPBatchRunner_GetPredictions(FILE* pOut, FILE* pErr, SQ_Model pModel, SQ_Prediction pPrediction);
int SQPBatchRunner_GetBatchModelInfo(FILE* pOut, FILE* pErr, SQ_Model pModel);
void SQPBatchRunner_GetResults(SQ_Model pModel, FILE* pOut, FILE* pErr, SQ_Prediction pHandle);
void SQPBatchRunner_GetAlignedResults(SQ_Model pModel, FILE* pOut, FILE* pErr, SQ_Prediction pHandle);
int SQPBatchRunner_GetModelParameters(SQ_Model pModel, FILE* pOut, FILE* pErr);
int SQPBatchRunner_GetAlignedParameters(SQ_Model pModel, FILE* pOut, FILE* pErr);
void PrintVectorData(SQ_VectorData pVectorData, FILE *pOut, FILE* pErr);

int IterateBatchProject(SQPBatchRunner* pObj, FILE* pOut, FILE* pErr);
int IterateContinousProject(SQPBatchRunner* pObj, FILE* pOut, FILE* pErr);
int CreateFakePredictionset(SQ_VariableVector hVariableVector, SQ_PreparePrediction pPreparePrediction, FILE* pErr);
int CreateFakeBatchPredictionset(int iModelIndex, SQ_VariableVector hVariableVector, SQ_PrepareBatchPrediction pPrepareBatchPrediction, FILE* pErr);

static char sszErrBuffer[1024];
#define ERRMSG(var, msg) sprintf(sszErrBuffer, "File: %s Line: %d : Message: %s\n", __FILE__, __LINE__, msg); var = sszErrBuffer

void PrintModelName(FILE* pOut, FILE* pErr, SQ_Model pModel)
{
   char* szErrString;
   char szModelName[100];

   if (SQ_FAILED(SQ_GetModelName(pModel, szModelName, 100)))
   {
      ERRMSG(szErrString, "GetModelName failed.");
      goto error_exit;
   }

   UTF8_printf(pOut, "Model name: %s\n", szModelName);
   return;
error_exit:
   PrintUTF8String(pErr, szErrString);
}

void PrintUTF8StringVector(SQ_StringVector pStringVector, FILE *pOut, FILE* pErr, const char szSeparator)
{
   char* szErrString;
   char strVal[1000];
   int iIter;
   int iNumStrings = 0;

   if (SQ_FAILED(SQ_GetNumStringsInVector(pStringVector, &iNumStrings)))
   {
      ERRMSG(szErrString, "SQ_GetNumStringsInVector failed.");
      goto error_exit;
   }

   for (iIter = 1; iIter <= iNumStrings; ++iIter)
   {
      /* Print the results tab separated to the file */
      if (SQ_FAILED(SQ_GetStringFromVector(pStringVector, iIter, strVal, 1000)))
      {
         ERRMSG(szErrString, "SQ_GetStringFromVector failed.");
         goto error_exit;
      }

      UTF8_printf(pOut, "%s%c", strVal, szSeparator);
   }
   PrintUTF8String(pOut, "\n");
   return;
error_exit:
   PrintUTF8String(pErr, szErrString);
}

void PrintVectorData(SQ_VectorData pVectorData, FILE *pOut, FILE* pErr)
{
   char* szErrString;
   char strVal[1000];
   float fVal;
   int iColIter;
   int iRowIter;
   int iRows = 0;
   int iCols = 0;
   SQ_FloatMatrix pMatrix = 0;
   SQ_StringVector pRowNames = 0;
   SQ_StringVector pColumnNames = 0;

   if (SQ_FAILED(SQ_GetDataMatrix(pVectorData, &pMatrix)))
   {
      ERRMSG(szErrString, "SQ_GetDataMatrix failed.");
      goto error_exit;
   }

   /* Most vectors are transposed so read number of columns to iRows. */
   if (SQ_FAILED(SQ_GetNumColumnsInFloatMatrix(pMatrix, &iRows)))
   {
      ERRMSG(szErrString, "SQ_GetNumColumnsInFloatMatrix failed.");
      goto error_exit;
   }
   if (SQ_FAILED(SQ_GetNumRowsInFloatMatrix(pMatrix, &iCols)))
   {
      ERRMSG(szErrString, "SQ_GetNumRowsInFloatMatrix failed.");
      goto error_exit;
   }
   if (SQ_FAILED(SQ_GetRowNames(pVectorData, &pColumnNames)))
   {
      ERRMSG(szErrString, "SQ_GetRowNames failed.");
      goto error_exit;
   }
   if (SQ_FAILED(SQ_GetColumnNames(pVectorData, &pRowNames)))
   {
      ERRMSG(szErrString, "SQ_GetColumnNames failed.");
      goto error_exit;
   }

   PrintUTF8StringVector(pColumnNames, pOut, pErr, '\t');
   for (iRowIter = 0; iRowIter < iRows; ++iRowIter)
   {
      /* Print the row name */
      if (SQ_FAILED(SQ_GetStringFromVector(pRowNames, iRowIter + 1, strVal, 1000)))
      {
         ERRMSG(szErrString, "GetModelName failed.");
         goto error_exit;
      }

      PrintUTF8String(pOut, strVal);
      PrintUTF8String(pOut, "\t");

      for (iColIter = 0; iColIter < iCols; ++iColIter)
      {
         /* Print the results tab separated to the file */
         if (SQ_FAILED(SQ_GetDataFromFloatMatrix(pMatrix, iColIter + 1, iRowIter + 1, &fVal)))
         {
            ERRMSG(szErrString, "SQ_GetDataFromFloatMatrix failed.");
            goto error_exit;
         }

         UTF8_printf(pOut, "%.8f\t", fVal);
      }
      PrintUTF8String(pOut, "\n");
   }
   PrintUTF8String(pOut, "\n");

   return;
error_exit:
   PrintUTF8String(pErr, szErrString);
}

int SQPBatchRunner_Init(SQPBatchRunner* pObj, const char* szUSPName)
{
   pObj->mszUSPName = (char*)strdup(szUSPName);

   pObj->mszFileBuffer = NULL;
   pObj->miFileNumCols = 0;
   pObj->miFileNumRows = 0;
   pObj->miPhaseColumn = 0;
   pObj->miPhases = 0;
   pObj->mpvecPhaseRows = NULL;
   pObj->miSizePhaseRows = 0;
   pObj->mpProjHandle = NULL;
   pObj->mstrVariableNames = NULL;

   return -1;
}

void SQPBatchRunner_Run(SQPBatchRunner* pObj, FILE* pOut, FILE* pErr)
{
   char* szErrString;
   char szProjectName[1000];
   int iIndexes = 0;
   SQ_Bool bIsBatchProject;
   SQ_BatchModel pBatchModel = NULL;
   SQ_Bool bValid;

   pObj->mpProjHandle = NULL;

   PrintUTF8String(pOut, "Path to usp file: ");
   PrintUTF8String(pOut, pObj->mszUSPName);
   PrintUTF8String(pOut, "\n");

   if (SQ_IsLicenseFileValid(&bValid) != SQ_E_OK)
   {
	   ERRMSG(szErrString, "Could not read license..");
	   goto error_exit;
   }
   else if (!bValid)
   {
	   ERRMSG(szErrString, "Invalid license..");
	   goto error_exit;
   }

   if (SQ_FAILED(SQ_OpenProject(pObj->mszUSPName, NULL/* Not a password protected project*/, &pObj->mpProjHandle)))
   {
      ERRMSG(szErrString, "Could not open project. SQ_OpenProject failed.");
      goto error_exit;
   }

   if (SQ_FAILED(SQ_GetIsBatchProject(pObj->mpProjHandle, &bIsBatchProject)))
   {
      ERRMSG(szErrString, "SQ_GetIsBatchProject failed.");
      goto error_exit;
   }
   pObj->mbIsBatchProject = (bIsBatchProject == SQ_True);

   /* Get the name of the project. */
   if (SQ_FAILED(SQ_GetProjectName(pObj->mpProjHandle, szProjectName, 1000)))
   {
      ERRMSG(szErrString, "GetProjectName failed.");
      goto error_exit;
   }

   PrintUTF8String(pOut, "Project name: ");
   PrintUTF8String(pOut, szProjectName);
   PrintUTF8String(pOut, "\n");

   /* Get the models in the project */
   if (pObj->mbIsBatchProject)
   {
      if (SQ_FAILED(SQ_GetBatchModel(pObj->mpProjHandle, 1, &pBatchModel)))
      {
         ERRMSG(szErrString, "SQ_GetBatchModel failed.");
         goto error_exit;
      }

      /* Get the number of models in the Batch Model. */
      if (SQ_FAILED(SQ_GetNumberOfBEM(pBatchModel, &iIndexes)))
      {
         ERRMSG(szErrString, "SQ_GetNumberOfBEM failed.");
         goto error_exit;
      }
      UTF8_printf(pOut, "Number of Batch Evolution models: %d\n\n", iIndexes);
   }
   else
   {
      /* Get the number of models in the project. */
      if (SQ_FAILED(SQ_GetNumberOfModels(pObj->mpProjHandle, &iIndexes)))
      {
         ERRMSG(szErrString, "SQ_GetNumberOfModels failed.");
         goto error_exit;
      }
      UTF8_printf(pOut, "Number of models: %d\n\n", iIndexes);
   }

   if (bIsBatchProject)
      IterateBatchProject(pObj, pOut, pErr);
   else
      IterateContinousProject(pObj, pOut, pErr);

   SQ_CloseProject(&pObj->mpProjHandle);

   return;

error_exit:
   PrintUTF8String(pErr, szErrString);
   SQ_CloseProject(&pObj->mpProjHandle);
}

void SQPBatchRunner_Destroy(SQPBatchRunner* pObj)
{
   free(pObj->mszUSPName);
   if(pObj->mszFileBuffer)
   {
      SQ_ClearStringVector(&pObj->mstrVariableNames);
      free(pObj->mszFileBuffer);
   }

   if (pObj->mpvecPhaseRows)
   {
      free(pObj->mpvecPhaseRows);
   }
}

int IterateContinousProject(SQPBatchRunner* pObj, FILE* pOut, FILE* pErr)
{
   char* szErrString;
   int iModelNumber;
   int iIndexes = 0;
   int iModelIndex;
   int iNumVariables;
   int iVariable;
   SQ_Prediction pPrediction = NULL;
   SQ_PreparePrediction pPreparePrediction = NULL;
   SQ_Bool bIsFitted;

   /* Data to SIMCA-Q.
   Go through all models.
   */
   if (SQ_FAILED(SQ_GetNumberOfModels(pObj->mpProjHandle, &iIndexes)))
   {
      ERRMSG(szErrString, "SQ_GetNumberOfModels failed.");
      goto error_exit;
   }

   for (iModelIndex = 1; iModelIndex <= iIndexes; ++iModelIndex)
   {

      /* Get the model number from the model index. */
      if (SQ_FAILED(SQ_GetModelNumberFromIndex(pObj->mpProjHandle, iModelIndex, &iModelNumber)))
      {
         ERRMSG(szErrString, "SQ_GetModelNumberFromIndex failed.");
         goto error_exit;
      }

      /* Get the model */
      if (iModelNumber > 0)
      {
         SQ_Model pModel = NULL;

         if (SQ_FAILED(SQ_GetModel(pObj->mpProjHandle, iModelNumber, &pModel)))
         {
            ERRMSG(szErrString, "SQ_GetModel failed.");
            goto error_exit;
         }

         if (SQ_FAILED(SQ_IsModelFitted(pModel, &bIsFitted)))
         {
            ERRMSG(szErrString, "IsModelFitted failed.");
            goto error_exit;
         }

         if (bIsFitted == SQ_False)
         {
            /* The model is not fitted, go to the next model. */
            PrintUTF8String(pOut, "Model is not fitted.\n");
         }
         else
         {
            SQ_VariableVector pVariableVector = NULL;

            PrintModelName(pOut, pErr, pModel);

            if (SQ_FAILED(SQ_GetPreparePrediction(pModel, &pPreparePrediction)))
            {
               ERRMSG(szErrString, "SQ_GetPreparePrediction failed.");
               goto error_exit;
            }
            if (SQ_FAILED(SQ_GetVariablesForPrediction(pPreparePrediction, &pVariableVector)))
            {
               ERRMSG(szErrString, "SQ_GetVariablesForPrediction failed.");
               goto error_exit;
            }

            if (SQ_FAILED(SQ_GetNumVariablesInVector(pVariableVector, &iNumVariables)))
            {
               ERRMSG(szErrString, "SQ_GetNumVariablesInVector failed.");
               goto error_exit;
            }

            /* Print all variables in the model */
            UTF8_printf(pOut, "Variables for model %d:\n", iModelNumber);
            for (iVariable = 1; iVariable <= iNumVariables; ++iVariable)
            {
               SQ_Variable oVariable = NULL;
               char szName[100];

               if (SQ_FAILED(SQ_GetVariableFromVector(pVariableVector, iVariable, &oVariable)))
               {
                  ERRMSG(szErrString, "SQ_GetVariableFromVector failed.");
                  goto error_exit;
               }
               if (SQ_FAILED(SQ_GetVariableName(oVariable, 1, szName, 100)))
               {
                  ERRMSG(szErrString, "SQ_GetVariableName failed.");
                  goto error_exit;
               }

               PrintUTF8String(pOut, szName);
               PrintUTF8String(pOut, "\t");
            }
            PrintUTF8String(pOut, "\n");

            /* Create a fake prediction set */
            CreateFakePredictionset(pVariableVector, pPreparePrediction, pErr);

            if (SQ_FAILED(SQ_GetPrediction(pPreparePrediction, &pPrediction)))
            {
               ERRMSG(szErrString, "SQ_GetPrediction failed.");
               goto error_exit;
            }

            SQPBatchRunner_GetPredictions(pOut, pErr, pModel, pPrediction);

         }
      }
   }

   /* Clean up */
   SQ_ClearPreparePrediction(&pPreparePrediction);
   SQ_ClearPrediction(&pPrediction);
   return 1;
error_exit:
   /* Clean up */
   SQ_ClearPreparePrediction(&pPreparePrediction);
   SQ_ClearPrediction(&pPrediction);
   PrintUTF8String(pErr, szErrString);
   return 0;
}

int IterateBatchProject(SQPBatchRunner* pObj, FILE* pOut, FILE* pErr)
{
   char* szErrString;
   int iModelNumber;
   int iNumVariables;
   int iVariable;
   int iNumBatchModels;
   int iBatchModel;
   int iPhase;
   int iBLMIx;
   int iNumBEM;
   int iNumBLM;
   SQ_Bool bIsFitted;
   SQ_BatchPrediction pPrediction = NULL;
   SQ_Prediction pBEMPrediction = NULL;
   SQ_PrepareBatchPrediction pPrepareBatchPrediction = NULL;
   SQ_BatchModel pBatchModel = NULL;
   SQ_Model pBLModel = NULL;

   SQ_GetNumberOfBatchModels(pObj->mpProjHandle, &iNumBatchModels);

   for (iBatchModel = 1; iBatchModel <= iNumBatchModels; ++iBatchModel)
   {
      if (SQ_FAILED(SQ_GetBatchModel(pObj->mpProjHandle, iBatchModel, &pBatchModel)))
      {
         ERRMSG(szErrString, "SQ_GetBatchModel failed.");
         goto error_exit;
      }

      SQ_GetNumberOfBEM(pBatchModel, &iNumBEM);
      SQ_GetNumberOfBLM(pBatchModel, &iNumBLM);
      if (iNumBLM == 0) /* Only predict those who have a batch level model. */
         continue;

      if (SQ_FAILED(SQ_GetBatchLevelModelNumber(pBatchModel, 1, &iModelNumber)))
      {
         ERRMSG(szErrString, "SQ_GetBatchLevelModelNumber failed.");
         goto error_exit;
      }
      if (SQ_FAILED(SQ_GetBatchLevelModel(pBatchModel, iModelNumber, &pBLModel)))
      {
         ERRMSG(szErrString, "SQ_GetBatchLevelModel failed.");
         goto error_exit;
      }

      /* Print Batch level model info */
      for (iBLMIx = 1; iBLMIx <= iNumBLM; ++iBLMIx)
      {
         SQ_GetBatchLevelModelNumber(pBatchModel, iBLMIx, &iModelNumber);
         SQ_GetBatchLevelModel(pBatchModel, iModelNumber, &pBLModel);

         SQ_IsModelFitted(pBLModel, &bIsFitted);
         if (bIsFitted != SQ_True)
         {
            /* The model is not fitted, go to the next model. */
            PrintUTF8String(pOut, "Model is not fitted.\n");
         }
         else
         {

            SQ_VariableVector pVariableVector = NULL;

            /* Get the model name. */
            PrintModelName(pOut, pErr, pBLModel);

            /* Get the prepare predictionset object for this batch level model */
            if (SQ_FAILED(SQ_GetPrepareBatchPrediction(pBLModel, &pPrepareBatchPrediction)))
            {
               ERRMSG(szErrString, "SQ_GetPrepareBatchPrediction failed.");
               goto error_exit;
            }

            /* Loop through all phases */
            for (iPhase = 1; iPhase <= iNumBEM; ++iPhase)
            {
               SQ_GetBatchEvolutionModelNumber(pBatchModel, iPhase, &iModelNumber);
               SQ_GetVariablesForBatchPrediction(pPrepareBatchPrediction, iPhase, &pVariableVector);
               if (SQ_FAILED(SQ_GetNumVariablesInVector(pVariableVector, &iNumVariables)))
               {
                  ERRMSG(szErrString, "SQ_GetNumVariablesInVector failed.");
                  goto error_exit;
               }

               /* Print all variables in the model */
               UTF8_printf(pOut, "Variables for model %d:\n", iModelNumber);
               for (iVariable = 1; iVariable <= iNumVariables; ++iVariable)
               {
                  SQ_Variable oVariable = NULL;
                  char szName[100];

                  if (SQ_FAILED(SQ_GetVariableFromVector(pVariableVector, iVariable, &oVariable)))
                  {
                     ERRMSG(szErrString, "SQ_GetVariableFromVector failed.");
                     goto error_exit;
                  }
                  if (SQ_FAILED(SQ_GetVariableName(oVariable, 1, szName, 100)))
                  {
                     ERRMSG(szErrString, "SQ_GetVariableName failed.");
                     goto error_exit;
                  }
                  PrintUTF8String(pOut, szName);
                  PrintUTF8String(pOut, "\t");
               }
               PrintUTF8String(pOut, "\n");

               /* Create the fake prediction set */
               CreateFakeBatchPredictionset(iPhase, pVariableVector, pPrepareBatchPrediction, pErr);
            }

            if (SQ_FAILED(SQ_GetBatchPrediction(pPrepareBatchPrediction, &pPrediction)))
            {
               ERRMSG(szErrString, "SQ_GetPrediction failed.");
               goto error_exit;
            }

            SQPBatchRunner_GetBatchModelInfo(pOut, pErr, pBLModel);

            /* Get batch evolution prediction */
            for (iPhase = 1; iPhase <= iNumBEM; ++iPhase)
            {
               SQ_GetBatchEvolutionModelNumber(pBatchModel, iPhase, &iModelNumber);
               SQ_GetBatchEvolutionPrediction(pPrediction, iModelNumber, &pBEMPrediction);
               SQPBatchRunner_GetAlignedResults(pBLModel, pOut, pErr, pBEMPrediction);
            }
         }
      }
   }

   /* Clean up */
   SQ_ClearPrepareBatchPrediction(&pPrepareBatchPrediction);
   SQ_ClearBatchPrediction(&pPrediction);

   return 1;
error_exit:
   /* Clean up */
   SQ_ClearPrepareBatchPrediction(&pPrepareBatchPrediction);
   SQ_ClearBatchPrediction(&pPrediction);
   PrintUTF8String(pErr, szErrString);
   return 0;
}

int CreateFakePredictionset(SQ_VariableVector hVariableVector, SQ_PreparePrediction pPreparePrediction, FILE* pErr)
{
   int iIx;
   int iNumVariables;
   int iObs;
   int iNumObs = 6;    /* Number of observations to predict.*/
   int iNumQualSettings = 0;
   float fRawData = 0;
   char* szErrString;
   SQ_Variable pVariable = NULL;
   SQ_Bool bIsQual;
   SQ_StringVector oQualSettings = NULL;
   
   SQ_GetNumVariablesInVector(hVariableVector, &iNumVariables);

   for (iIx = 1; iIx <= iNumVariables; ++iIx)
   {
      if (SQ_FAILED(SQ_GetVariableFromVector(hVariableVector, iIx, &pVariable)))
      {
         ERRMSG(szErrString, "SQ_GetVariableFromVector failed.");
         goto error_exit;
      }
      if (SQ_FAILED(SQ_IsQualitative(pVariable, &bIsQual)))
      {
         ERRMSG(szErrString, "SQ_IsQualitative failed.");
         goto error_exit;
      }

      /* Get all qualitative settings */
      if (bIsQual == SQ_True)
      {
         if (SQ_FAILED(SQ_GetQualitativeSettings(pVariable, &oQualSettings)))
         {
            ERRMSG(szErrString, "SQ_GetQualitativeSettings failed.");
            goto error_exit;
         }
         if (SQ_FAILED(SQ_GetNumStringsInVector(oQualSettings, &iNumQualSettings)))
         {
            ERRMSG(szErrString, "SQ_GetNumStringsInVector failed.");
            goto error_exit;
         }
      }
      for(iObs = 1; iObs <= iNumObs; ++iObs)
      {
         if (bIsQual == SQ_True)
         {
            /* This is a qualitative variable, rand a setting from the variable. */
            int iSetting = (rand() % (iNumQualSettings-1) + 1); /* Rand a number from 1 to iNumQualSettings. */
            char szSetting[100];
            if (SQ_FAILED(SQ_GetStringFromVector(oQualSettings, iSetting, szSetting, 100)))
            {
               ERRMSG(szErrString, "SQ_GetStringFromVector failed.");
               goto error_exit;
            }
            
            if (SQ_FAILED(SQ_SetQualitativeData(pPreparePrediction, iObs, iIx, szSetting)))
            {
               ERRMSG(szErrString, "SQ_SetQualitativeData failed.");
               goto error_exit;
            }
         }
         else
         {
            /* Only insert [1,2,3,...,iNumVariables] for this sample, 
            Instead you can for example read the data from a file or database.
            */
            if (SQ_FAILED(SQ_SetQuantitativeData(pPreparePrediction, iObs, iIx, ++fRawData)))
            {
               ERRMSG(szErrString, "SQ_SetQuantitativeData failed.");
               goto error_exit;
            }
         }
      }
   }

   SQ_ClearStringVector(&oQualSettings);
   return 1;
error_exit:
   SQ_ClearStringVector(&oQualSettings);
   PrintUTF8String(pErr, szErrString);
   return 0;
}

int CreateFakeBatchPredictionset(int iModelIndex, SQ_VariableVector hVariableVector, SQ_PrepareBatchPrediction pPrepareBatchPrediction, FILE* pErr)
{
   int iIx;
   int iNumVariables;
   int iObs;
   int iNumObs = 6;    /* Number of observations to predict.*/
   int iNumQualSettings = 0;
   float fRawData = 0;
   char* szErrString;
   SQ_Variable pVariable = NULL;
   SQ_Bool bIsQual;
   SQ_StringVector oQualSettings = NULL;

   SQ_GetNumVariablesInVector(hVariableVector, &iNumVariables);

   for (iIx = 1; iIx <= iNumVariables; ++iIx)
   {
      if (SQ_FAILED(SQ_GetVariableFromVector(hVariableVector, iIx, &pVariable)))
      {
         ERRMSG(szErrString, "SQ_GetVariableFromVector failed.");
         goto error_exit;
      }
      if (SQ_FAILED(SQ_IsQualitative(pVariable, &bIsQual)))
      {
         ERRMSG(szErrString, "SQ_IsQualitative failed.");
         goto error_exit;
      }

      if (bIsQual == SQ_True)
      {
         if (SQ_FAILED(SQ_GetQualitativeSettings(pVariable, &oQualSettings)))
         {
            ERRMSG(szErrString, "SQ_GetQualitativeSettings failed.");
            goto error_exit;
         }
         if (SQ_FAILED(SQ_GetNumStringsInVector(oQualSettings, &iNumQualSettings)))
         {
            ERRMSG(szErrString, "SQ_GetNumStringsInVector failed.");
            goto error_exit;
         }
      }
      for(iObs = 1; iObs <= iNumObs; ++iObs)
      {
         if (bIsQual == SQ_True)
         {
            /* This is a qualitative variable, rand a setting from the variable. */
            int iSetting = (rand() % (iNumQualSettings-1) + 1); /* Rand a number from 1 to iNumQualSettings. */
            char szSetting[100];
            if (SQ_FAILED(SQ_GetStringFromVector(oQualSettings, iSetting, szSetting, 100)))
            {
               ERRMSG(szErrString, "SQ_GetStringFromVector failed.");
               goto error_exit;
            }

            if (SQ_FAILED(SQ_SetQualitativeBatchData(pPrepareBatchPrediction, iModelIndex, iObs, iIx, szSetting)))
            {
               ERRMSG(szErrString, "SQ_SetQualitativeBatchData failed.");
               goto error_exit;
            }

         }
         else
         {
            /* Only insert [1,2,3,...,iNumVariables] for this sample, 
            Instead you can for example read the data from a file or database.
            */
            if (SQ_FAILED(SQ_SetQuantitativeBatchData(pPrepareBatchPrediction, iModelIndex, iObs, iIx, ++fRawData)))
            {
               ERRMSG(szErrString, "SQ_SetQuantitativeBatchData failed.");
               goto error_exit;
            }
         }
      }
   }

   SQ_ClearStringVector(&oQualSettings);
   return 1;
error_exit:
   SQ_ClearStringVector(&oQualSettings);
   PrintUTF8String(pErr, szErrString);

   return 0;
}

int SQPBatchRunner_GetModelParameters(SQ_Model hModel, FILE* pOut, FILE* pErr)
{
   char* szErrString;
   SQ_VectorData pVectorData = NULL;
   SQ_ModelType eModelType;
   int iNumComp = 0;  /* Number of components. */

   /* Get Number of components. */
   if (SQ_FAILED(SQ_GetNumberOfComponents(hModel, &iNumComp)))
   {
      ERRMSG(szErrString,  "SQ_GetNumberOfComponents failed.");
      goto error_exit;
   } 

   /* Get T for all components in the model. */
   if (SQ_FAILED(SQ_GetT(hModel, NULL /*pnComponents*/, &pVectorData)))
   {
      ERRMSG(szErrString, "GetT failed.");
      goto error_exit;
   }
   else
   {
      PrintUTF8String(pOut, "T:\n");
      PrintVectorData(pVectorData, pOut, pErr);
   }

   /* Get DModX for all components in the model. */
   if (SQ_FAILED(SQ_GetDModX(hModel, NULL /*pnComponents*/, SQ_Normalized_Default, SQ_ModelingPowerWeighted_Default, &pVectorData)))
   {
      ERRMSG(szErrString, "GetDModX failed.");
      goto error_exit;
   }
   else
   {
      PrintUTF8String(pOut, "DModX:\n");
      PrintVectorData(pVectorData, pOut, pErr);
   }

   if (SQ_FAILED(SQ_GetModelType(hModel, &eModelType)))
   {
      ERRMSG(szErrString,  "SQ_GetModelType failed.");
      goto error_exit;
   }

   if ((eModelType == SQ_PLS_Class) || (eModelType == SQ_PLS) || (eModelType == SQ_PLS_DA) || (eModelType == SQ_OPLS))
   {
      /* Get C for all components in the model. */
      if (SQ_FAILED(SQ_GetC(hModel, NULL /*pComponents*/, &pVectorData)))
      {
         ERRMSG(szErrString,  "SQ_GetModelC failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "C:\n");
         PrintVectorData(pVectorData, pOut, pErr);
      }
   }
   else
   {
      /* PCA */
      /* Get Q2VX */
      if (SQ_FAILED(SQ_GetQ2VX(hModel, NULL, &pVectorData)))
      {
         ERRMSG(szErrString,  "SQ_GetModelQ2VX failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "Q2VX:\n");
         PrintVectorData(pVectorData, pOut, pErr);
      }
   }

   SQ_ClearVectorData(&pVectorData);

   return 1;
error_exit:
   SQ_ClearVectorData(&pVectorData);
   PrintUTF8String(pErr, szErrString);
   return 0;
}

int SQPBatchRunner_GetAlignedParameters(SQ_Model hModel, FILE* pOut, FILE* pErr)
{
   char* szErrString;
   int iNumComp = 0;  /* Number of components. */
   SQ_ModelType eModelType;
   SQ_VectorData oVecData = NULL;

   /* Get Number of components. */
   if (SQ_FAILED(SQ_GetNumberOfComponents(hModel, &iNumComp)))
   {
      ERRMSG(szErrString,  "GetModelNumberOfComponents failed.");
      goto error_exit;
   } 

   if (SQ_FAILED(SQ_GetModelType(hModel, &eModelType)))
   {
      ERRMSG(szErrString,  "SQ_GetModelType failed.");
      goto error_exit;
   }

   if ((eModelType == SQ_PLS_Class) || (eModelType == SQ_PLS) || (eModelType == SQ_PLS_DA) || (eModelType == SQ_OPLS))
   {
      if (SQ_FAILED(SQ_GetAlignedT2Range(hModel, 1 /*iCompFrom*/, -1 /*iCompTo*/, NULL, &oVecData)))
      {
         ERRMSG(szErrString,  "SQ_GetModelAlignedT2Range failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "ModelAlignedT2Range:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }
      SQ_ClearVectorData(&oVecData);
   }

   return 1;
error_exit:
   SQ_ClearVectorData(&oVecData);
   PrintUTF8String(pErr, szErrString);
   return 0;
}             

void SQPBatchRunner_GetResults(SQ_Model hModel, FILE* pOut, FILE* pErr, SQ_Prediction pHandle) 
{
   char* szErrString;
   int iNumComp = 0;
   SQ_ModelType eModelType;
   SQ_VectorData oVecData = NULL;

   if (SQ_FAILED(SQ_GetNumberOfComponents(hModel, &iNumComp)))
   {
      ERRMSG(szErrString,  "GetModelNumberOfComponents failed.");
      goto error_exit;
   }
   /* Get XVarPS */
   if (SQ_FAILED(SQ_GetXVarPS(pHandle, SQ_Unscaled_False, SQ_Backtransformed_True, NULL /*pColumnXList*/, &oVecData)))
   {
      ERRMSG(szErrString,  "SQ_GetXVarPS failed.");
      goto error_exit;
   }
   else
   {
      PrintUTF8String(pOut, "XVarPS:\n");
      PrintVectorData(oVecData, pOut, pErr);
   }

   if (SQ_FAILED(SQ_GetContributionsScorePSSingleWeight(pHandle, 0 /*iObs1Ix*/, 1/*iObs2Ix*/, SQ_Weight_Normalized, iNumComp, 1 /*iYVar*/, SQ_Reconstruct_False, &oVecData)))
   {
      ERRMSG(szErrString,  "SQP_GetPredictedContributionsSSW failed.");
      goto error_exit;
   }
   else
   {
      PrintUTF8String(pOut, "Predicted Contribution SSW:\n");
      PrintVectorData(oVecData, pOut, pErr);
   }

   /* Get the predicted DModX. */
   if (SQ_FAILED(SQ_GetDModXPS(pHandle, NULL /*pnComponentList*/, 1 /*bNormalized*/, 0 /*bModelingPowerWeighted*/, &oVecData)))
   {
      ERRMSG(szErrString,  "GetPredictedDModX failed.");
      goto error_exit;
   }
   else
   {

      PrintUTF8String(pOut, "\nDModXPS:\n");
      PrintVectorData(oVecData, pOut, pErr);
   }

   if (iNumComp > 0)
   {   
      /* Get TPS */
      if (SQ_FAILED(SQ_GetTPS(pHandle, NULL /*pnComponentList*/, &oVecData)))
      {
         ERRMSG(szErrString,  "GetPredictedT failed.");
         goto error_exit;
      }
      else
      {

         PrintUTF8String(pOut, "\nTPS:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }

      /* Get T2RangePS */
      if (SQ_FAILED(SQ_GetT2RangePS(pHandle, 1 /*nCompFrom*/, iNumComp /*nCompTo*/, &oVecData)))
      {
         ERRMSG(szErrString,  "GetPredictedT2Range failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "\nT2RangePS:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }
   }

   /* Retrieve the model type. */
   if (SQ_FAILED(SQ_GetModelType(hModel, &eModelType)))
   {
      ERRMSG(szErrString,  "GetModelType failed.");
      goto error_exit;
   }
   if ((iNumComp > 0) && ((eModelType == SQ_PLS_Class) || (eModelType == SQ_PLS) || (eModelType == SQ_PLS_DA) || (eModelType == SQ_OPLS)))
   {
      /*Get YPredPS */
      if (SQ_FAILED(SQ_GetYPredPS(pHandle, iNumComp, 0 /*bUnscaled*/, 0 /*bBackTransformed*/, NULL /*pnColumnYIndexes*/, &oVecData)))
      {
         ERRMSG(szErrString,  "GetPredictedY failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "YPredPS:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }
   }  

   /* Remove the vector data before going out of scope */
   SQ_ClearVectorData(&oVecData); 

   return;
error_exit:
   SQ_ClearVectorData(&oVecData); 
   PrintUTF8String(pErr, szErrString);
} 

void SQPBatchRunner_GetAlignedResults(SQ_Model hModel, FILE* pOut, FILE* pErr, SQ_Prediction pHandle)
{
   char* szErrString;
   int iNumComp = 0;
   SQ_ModelType eModelType;
   SQ_VectorData oVecData = NULL;

   if (SQ_FAILED(SQ_GetNumberOfComponents(hModel, &iNumComp)))
   {
      ERRMSG(szErrString,  "GetModelNumberOfComponents failed.");
      goto error_exit;
   }

   /* Retrieve the model type. */
   if (SQ_FAILED(SQ_GetModelType(hModel, &eModelType)))
   {
      ERRMSG(szErrString,  "GetModelType failed.");
      goto error_exit;
   }

   if ((iNumComp > 0) && ((eModelType == SQ_PLS_Class) || (eModelType == SQ_PLS) || (eModelType == SQ_PLS_DA) || (eModelType == SQ_OPLS)))
   {
      PrintUTF8String(pOut, "**************************************************************************\n");
      PrintUTF8String(pOut, "          Predicted data\n");
      PrintUTF8String(pOut, "**************************************************************************\n");

      /* This is a PLS model.*/
      if (SQ_FAILED(SQ_GetAlignedT2RangePS(pHandle, 1 /*iCompFrom*/, -1 /*iCompTo*/, &oVecData)))
      {
         ERRMSG(szErrString,  "GetPredictedAlignedT2Range failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "PredictedAlignedT2Range:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }
      if (SQ_FAILED(SQ_GetAlignedTimeMaturityPS(pHandle, &oVecData)))
      {
         ERRMSG(szErrString,  "GetPredictedAlignedTimeMaturity failed.");
         goto error_exit;
      }
      else
      {
         PrintUTF8String(pOut, "PredictedAlignedTimeMaturity:\n");
         PrintVectorData(oVecData, pOut, pErr);
      }

      SQ_ClearVectorData(&oVecData);
   }

   return;
error_exit:
   PrintUTF8String(pErr, szErrString);

   SQ_ClearVectorData(&oVecData);
}

int SQPBatchRunner_GetPredictions(FILE* pOut, FILE* pErr, SQ_Model pModel, SQ_Prediction pPrediction)
{
   PrintModelName(pOut, pErr, pModel);
   PrintUTF8String(pOut, "**************************************************************************\n");
   PrintUTF8String(pOut, "          Data from the SIMCA model\n");
   PrintUTF8String(pOut, "**************************************************************************\n");
   SQPBatchRunner_GetModelParameters(pModel,pOut, pErr);
   PrintUTF8String(pOut, "**************************************************************************\n");
   PrintUTF8String(pOut, "          Predicted data\n");
   PrintUTF8String(pOut, "**************************************************************************\n");
   SQPBatchRunner_GetResults(pModel, pOut, pErr, pPrediction);
   return 1;
}

int SQPBatchRunner_GetBatchModelInfo(FILE* pOut, FILE* pErr, SQ_Model pModel)
{
   PrintModelName(pOut, pErr, pModel);
   PrintUTF8String(pOut, "**************************************************************************\n");
   PrintUTF8String(pOut, "          Data from the SIMCA model\n");
   PrintUTF8String(pOut, "**************************************************************************\n");
   SQPBatchRunner_GetModelParameters(pModel, pOut, pErr);
   return 1;
}

